<?php
/*
 * Plugin Name: Appuz Appointments
 * Description: Sistema de agendamento de sessões para terapeutas.
 * Version: 4.9
 * Author: Renan Daniel
 * Text Domain: appuz
 */

if (!defined('ABSPATH')) exit; // Exit if accessed directly

define('APPUZ_PLUGIN_VERSION', 'v4.9');

// Verifica se o WooCommerce está ativo
if (!in_array('woocommerce/woocommerce.php', apply_filters('active_plugins', get_option('active_plugins')))) {
    add_action('admin_notices', function() {
        echo '<div class="error"><p>' . __('Appuz Appointments requires WooCommerce to be installed and active.', 'appuz') . '</p></div>';
    });
    return;
}

// Função para criar tabelas no banco de dados
function appuz_create_tables() {
    global $wpdb;

    $charset_collate = $wpdb->get_charset_collate();

    $appointments_table = $wpdb->prefix . 'appuz_appointments';
    $sql_appointments = "CREATE TABLE IF NOT EXISTS $appointments_table (
        id bigint(20) NOT NULL AUTO_INCREMENT,
        patient_id bigint(20) NOT NULL,
        session_date datetime NOT NULL,
        status varchar(20) DEFAULT 'pending' NOT NULL,
        created_at datetime DEFAULT CURRENT_TIMESTAMP NOT NULL,
        PRIMARY KEY (id)
    ) $charset_collate;";

    $patients_table = $wpdb->prefix . 'appuz_patients';
    $sql_patients = "CREATE TABLE IF NOT EXISTS $patients_table (
        id bigint(20) NOT NULL AUTO_INCREMENT,
        name varchar(255) NOT NULL,
        email varchar(255) NOT NULL,
        phone varchar(50),
        created_at datetime DEFAULT CURRENT_TIMESTAMP NOT NULL,
        PRIMARY KEY (id)
    ) $charset_collate;";

    require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
    dbDelta($sql_appointments);
    dbDelta($sql_patients);
}
register_activation_hook(__FILE__, 'appuz_create_tables');

// Função para redirecionar apenas administradores após o login
function redirecionar_admin_apos_login($redirect_to, $request, $user) {
    // Verifica se o usuário é um administrador
    if (isset($user->roles) && in_array('administrator', $user->roles)) {
        // Substitua 'your-plugin-page' pela slug da página do plugin
        $pagina_plugin = admin_url('admin.php?page=appuz_appointments');
        return $pagina_plugin;
    }
    return $redirect_to; // Mantém o redirecionamento padrão para outros usuários
}
add_filter('login_redirect', 'redirecionar_admin_apos_login', 10, 3);

// Função para adicionar o botão flutuante do WhatsApp
function appuz_whatsapp_floating_button() {
    ?>
    <style>
        /* Estilo do botão flutuante do WhatsApp */
        .whatsapp-float-button {
            position: fixed;
            bottom: 20px;
            right: 20px;
            width: 50px;
            height: 50px;
            background-color: #25D366;
            color: white;
            border-radius: 50%;
            display: flex;
            justify-content: center;
            align-items: center;
            box-shadow: 0px 4px 10px rgba(0, 0, 0, 0.2);
            cursor: pointer;
            z-index: 1000;
        }
        .whatsapp-float-button img {
            width: 30px;
            height: 30px;
        }
    </style>
    
    <!-- Botão flutuante do WhatsApp -->
    <a href="https://wa.me/" target="_blank" class="whatsapp-float-button" title="Converse conosco no WhatsApp">
        <img src="https://upload.wikimedia.org/wikipedia/commons/5/5e/WhatsApp_icon.png" alt="WhatsApp Icon">
    </a>
    <?php
}
add_action('wp_footer', 'appuz_whatsapp_floating_button');

function appuz_create_availability_table() {
    global $wpdb;
    $table_name = $wpdb->prefix . 'disponibilidadeappus';
    $charset_collate = $wpdb->get_charset_collate();

    $sql = "CREATE TABLE IF NOT EXISTS $table_name (
        id bigint(20) NOT NULL AUTO_INCREMENT,
        day_of_week varchar(20) NOT NULL,
        off_hours text NOT NULL,
        PRIMARY KEY (id)
    ) $charset_collate;";

    require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
    dbDelta($sql);
}
register_activation_hook(__FILE__, 'appuz_create_availability_table');


// Função para criar a tabela appuz_temp_appointments para armazenar dados temporários de agendamento
function appuz_create_temp_appointments_table() {
    global $wpdb;
    $table_name = $wpdb->prefix . 'appuz_temp_appointments';
    $charset_collate = $wpdb->get_charset_collate();

    $sql = "CREATE TABLE IF NOT EXISTS $table_name (
        id bigint(20) NOT NULL AUTO_INCREMENT,
        order_id bigint(20) NOT NULL,
        patient_id bigint(20) NOT NULL,
        session_date datetime NOT NULL,
        PRIMARY KEY (id)
    ) $charset_collate;";

    require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
    dbDelta($sql);
}
register_activation_hook(__FILE__, 'appuz_create_temp_appointments_table');

// Função para criar a tabela 'horasdetrabalho'
function appuz_create_horasdetrabalho_table() {
    global $wpdb;
    $table_name = $wpdb->prefix . 'horasdetrabalho';
    $charset_collate = $wpdb->get_charset_collate();

    $sql = "CREATE TABLE IF NOT EXISTS $table_name (
        id bigint(20) NOT NULL AUTO_INCREMENT,
        hour_time varchar(5) NOT NULL,
        PRIMARY KEY (id),
        UNIQUE KEY hour_time (hour_time)
    ) $charset_collate;";

    require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
    dbDelta($sql);
}
register_activation_hook(__FILE__, 'appuz_create_horasdetrabalho_table');

// Página de Configuração de Disponibilidade
function appuz_availability_page() {
    global $wpdb;
    $table_disponibilidade = $wpdb->prefix . 'disponibilidadeappus';
    $table_horas_trabalho = $wpdb->prefix . 'horasdetrabalho';

    // Salvar as configurações se o formulário for enviado
    if (isset($_POST['save_availability'])) {
        $days_off = isset($_POST['days_off']) ? $_POST['days_off'] : [];
        $unavailable_times = isset($_POST['unavailable_times']) ? sanitize_text_field($_POST['unavailable_times']) : '';

        // Salva nas opções do WordPress (mantém compatibilidade)
        update_option('appuz_days_off', $days_off);
        update_option('appuz_unavailable_times', $unavailable_times);

        // Limpa os registros antigos nas tabelas personalizadas
        $wpdb->query("DELETE FROM $table_disponibilidade");
        $wpdb->query("DELETE FROM $table_horas_trabalho");

        // Insere os dias de folga e horários indisponíveis na tabela de disponibilidade
        foreach ($days_off as $day) {
            $wpdb->insert($table_disponibilidade, [
                'day_of_week' => $day,
                'off_hours' => maybe_serialize(explode(',', $unavailable_times)),
            ]);
        }

        // Insere os horários indisponíveis na tabela horasdetrabalho
        $unavailable_hours = explode(',', $unavailable_times);
        foreach ($unavailable_hours as $hour_time) {
            $wpdb->insert($table_horas_trabalho, [
                'hour_time' => sanitize_text_field(trim($hour_time)),
            ]);
        }

        echo '<div class="notice notice-success"><p>' . __('Configurações de disponibilidade salvas com sucesso!', 'appuz') . '</p></div>';
    }

    // Recuperar configurações salvas tanto das opções do WordPress quanto da tabela personalizada
    $saved_days_off = get_option('appuz_days_off', []);
    $saved_unavailable_times = get_option('appuz_unavailable_times', '');

    // Dados da tabela personalizada
    $availability_data = $wpdb->get_results("SELECT * FROM $table_disponibilidade", OBJECT_K);

    // Formulário de configuração
    echo '<div class="wrap">';
    echo '<h1>' . __('Configurar Dias de Folga e Horários Indisponíveis', 'appuz') . '</h1>';
    echo '<form method="post" action="">';

    // Seleção de dias de folga
    echo '<h2>' . __('Dias de Folga', 'appuz') . '</h2>';
    $days_of_week = [
        'sunday' => __('Domingo', 'appuz'),
        'monday' => __('Segunda-feira', 'appuz'),
        'tuesday' => __('Terça-feira', 'appuz'),
        'wednesday' => __('Quarta-feira', 'appuz'),
        'thursday' => __('Quinta-feira', 'appuz'),
        'friday' => __('Sexta-feira', 'appuz'),
        'saturday' => __('Sábado', 'appuz')
    ];
    foreach ($days_of_week as $day_key => $day_label) {
        $is_day_off = in_array($day_key, $saved_days_off) || isset($availability_data[$day_key]);
        echo '<label><input type="checkbox" name="days_off[]" value="' . esc_attr($day_key) . '" ' . checked($is_day_off, true, false) . '> ' . esc_html($day_label) . '</label><br>';
    }

    // Configuração de horários indisponíveis
    echo '<h2>' . __('Horários Indisponíveis', 'appuz') . '</h2>';
    echo '<p>' . __('Insira os horários em formato 24 horas, separados por vírgulas (ex: 13:00-15:00, 18:00-19:00).', 'appuz') . '</p>';

    // Exibir horários indisponíveis da tabela personalizada, se houver
    $displayed_times = $saved_unavailable_times ?: (isset($availability_data['off_hours']) ? implode(',', maybe_unserialize($availability_data['off_hours']->off_hours)) : '');
    echo '<input type="text" name="unavailable_times" value="' . esc_attr($displayed_times) . '" class="widefat">';

    echo '<br><br><button type="submit" name="save_availability" class="button button-primary">' . __('Salvar Configurações', 'appuz') . '</button>';
    echo '</form>';
    echo '</div>';
}

// Função para adicionar campos de agendamento no checkout do WooCommerce
function appuz_add_calendar_to_checkout($checkout) {
    if (WC()->cart->is_empty()) {
        return;
    }

    $cart = WC()->cart;
    $total_quantity = 0;
    foreach ($cart->get_cart() as $cart_item) {
        $total_quantity += $cart_item['quantity'];
    }

    if ($total_quantity > 0) {
        echo '<div id="appuz-appointment-calendar" style="padding: 20px; border: 1px solid #ddd; margin-top: 20px; border-radius: 8px; background: #f9f9f9;">';
        echo '<h3 style="font-weight: bold; margin-bottom: 10px;">' . __('Escolha as Datas e Horários das Sessões', 'appuz') . '</h3>';
        echo '<p style="margin-bottom: 15px; font-size: 14px; color: #555;">' . sprintf(__('Você comprou %d horas. Por favor, selecione %d horários distintos.', 'appuz'), $total_quantity, $total_quantity) . '</p>';

        for ($i = 1; $i <= $total_quantity; $i++) {
            echo '<p><label for="appuz_session_date_' . $i . '" style="font-weight: 600;">' . sprintf(__('Data %d:', 'appuz'), $i) . '</label>';
            echo '<input type="date" name="appuz_session_dates[' . $i . '][date]" id="appuz_session_date_' . $i . '" required style="margin-bottom: 10px; width: 100%; padding: 8px; border-radius: 4px; border: 1px solid #ccc;">';

            echo '<label for="appuz_session_time_' . $i . '" style="display: block; font-weight: 600; margin-top: 10px;">' . __('Horário:', 'appuz') . '</label>';
            echo '<select name="appuz_session_dates[' . $i . '][time]" id="appuz_session_time_' . $i . '" required style="width: 100%; padding: 8px; border-radius: 4px; border: 1px solid #ccc;">';
            echo '<option value="">' . __('Escolha seu Horário', 'appuz') . '</option>'; // Opção padrão
            echo '</select>';
            echo '<div id="appuz_unavailable_message_' . $i . '" style="color: red; display: none; margin-top: 5px;"></div>';
            echo '<div id="appuz_success_message_' . $i . '" style="color: green; display: none; margin-top: 5px;"></div>';

            // Script para verificar disponibilidade e horários dinâmicos
            echo '<script>
                jQuery(document).ready(function($) {
                    $("#appuz_session_date_' . $i . '").on("change", function() {
                        var selectedDate = $(this).val();
                        $.ajax({
                            url: "' . admin_url('admin-ajax.php') . '",
                            type: "POST",
                            data: {
                                action: "appuz_check_date_availability",
                                date: selectedDate
                            },
                            success: function(response) {
                                if (response === "unavailable") {
                                    $("#appuz_unavailable_message_' . $i . '").text("' . __('Data indisponível. Por favor, escolha outra data.', 'appuz') . '").show();
                                    $("#appuz_session_time_' . $i . '").html("").hide();
                                    $("#appuz_success_message_' . $i . '").hide();
                                    $("#appuz_session_date_' . $i . '").val(""); 
                                } else {
                                    $("#appuz_unavailable_message_' . $i . '").hide();
                                    $("#appuz_session_time_' . $i . '").html(\'<option value="">' . __('Escolha seu Horário', 'appuz') . '</option>\' + response).show(); // Define a opção padrão e adiciona horários disponíveis
                                }
                            }
                        });
                    });

                    $("#appuz_session_time_' . $i . '").on("change", function() {
                        var selectedDate = $("#appuz_session_date_' . $i . '").val();
                        var selectedTime = $(this).val();
                        $.ajax({
                            url: "' . admin_url('admin-ajax.php') . '",
                            type: "POST",
                            data: {
                                action: "appuz_check_time_availability",
                                date: selectedDate,
                                time: selectedTime
                            },
                            success: function(response) {
                                if (response === "unavailable") {
                                    $("#appuz_unavailable_message_' . $i . '").text("' . __('Este horário já foi agendado. Escolha outro horário.', 'appuz') . '").show();
                                    $("#appuz_success_message_' . $i . '").hide();
                                } else {
                                    $("#appuz_success_message_' . $i . '").text("' . __('Tudo certo com a sua sessão. Pode prosseguir!', 'appuz') . '").show();
                                    $("#appuz_unavailable_message_' . $i . '").hide();
                                }
                            }
                        });
                    });
                });
            </script>';

            echo '</p>';
        }

        echo '</div>';
    }
}
add_action('woocommerce_after_checkout_billing_form', 'appuz_add_calendar_to_checkout');

// Função para salvar as datas e horários de sessão como meta do pedido
function appuz_save_session_dates_to_order_meta($order_id) {
    if (isset($_POST['appuz_session_dates']) && !empty($_POST['appuz_session_dates'])) {
        // Salvar as datas e horários selecionados como meta do pedido
        update_post_meta($order_id, 'appuz_session_dates', $_POST['appuz_session_dates']);
    }
}
add_action('woocommerce_checkout_update_order_meta', 'appuz_save_session_dates_to_order_meta');

// Função AJAX para verificar a disponibilidade da data e horários
function appuz_check_date_availability() {
    $selected_date = sanitize_text_field($_POST['date']);
    $available_times = appuz_get_available_times($selected_date);

    if ($available_times === 'unavailable') {
        echo 'unavailable';
    } else {
        foreach ($available_times as $time) {
            echo '<option value="' . esc_attr($time) . '">' . esc_html($time) . '</option>';
        }
    }
    wp_die();
}
add_action('wp_ajax_appuz_check_date_availability', 'appuz_check_date_availability');
add_action('wp_ajax_nopriv_appuz_check_date_availability', 'appuz_check_date_availability');

// Função AJAX para verificar se o horário está disponível para o agendamento
function appuz_check_time_availability() {
    global $wpdb;
    $selected_date = sanitize_text_field($_POST['date']);
    $selected_time = sanitize_text_field($_POST['time']);
    $datetime = $selected_date . ' ' . $selected_time . ':00';

    $appointment_exists = $wpdb->get_var($wpdb->prepare("SELECT COUNT(*) FROM {$wpdb->prefix}appuz_appointments WHERE session_date = %s AND status = 'confirmed'", $datetime));

    if ($appointment_exists > 0) {
        echo 'unavailable';
    } else {
        echo 'available';
    }
    wp_die();
}
add_action('wp_ajax_appuz_check_time_availability', 'appuz_check_time_availability');
add_action('wp_ajax_nopriv_appuz_check_time_availability', 'appuz_check_time_availability');

// Função para verificar se o horário já está reservado
function appuz_is_time_confirmed($selected_date, $selected_time) {
    global $wpdb;
    $table_name = $wpdb->prefix . 'appuz_appointments';
    $datetime = $selected_date . ' ' . $selected_time . ':00';

    $existing = $wpdb->get_var($wpdb->prepare("SELECT COUNT(*) FROM $table_name WHERE session_date = %s AND status = 'confirmed'", $datetime));
    return $existing > 0;
}

// Função para recuperar horários disponíveis com base nas configurações
function appuz_get_available_times($selected_date) {
    global $wpdb;
    $table_name = $wpdb->prefix . 'disponibilidadeappus';
    $day_of_week = strtolower(date('l', strtotime($selected_date)));

    // Verifica se o dia selecionado é um dia de folga
    $day_off = $wpdb->get_var($wpdb->prepare("SELECT COUNT(*) FROM $table_name WHERE day_of_week = %s", $day_of_week));
    if ($day_off > 0) {
        return 'unavailable';
    }

    // Recupera os horários indisponíveis para o dia selecionado
    $off_hours_row = $wpdb->get_row("SELECT off_hours FROM $table_name LIMIT 1");
    $off_hours = maybe_unserialize($off_hours_row->off_hours);

    // Define todos os horários em intervalos de 1 hora
    $all_times = [];
    for ($hour = 0; $hour < 24; $hour++) {
        $all_times[] = str_pad($hour, 2, '0', STR_PAD_LEFT) . ':00';
    }

    // Remove os horários indisponíveis
    foreach ($off_hours as $interval) {
        list($start, $end) = explode('-', $interval);
        foreach ($all_times as $key => $time) {
            if ($time >= $start && $time <= $end) {
                unset($all_times[$key]);
            }
        }
    }

    // Recupera os horários já reservados para a data
    $reserved_times = $wpdb->get_col($wpdb->prepare(
        "SELECT DATE_FORMAT(session_date, '%%H:%%i') FROM {$wpdb->prefix}appuz_appointments 
         WHERE DATE(session_date) = %s AND status = 'confirmed'",
        $selected_date
    ));

    // Remove horários já reservados da lista de horários disponíveis
    $available_times = array_diff($all_times, $reserved_times);

    return array_values($available_times);
}


// Função para adicionar o CSS personalizado
function appuz_enqueue_styles() {
    echo '<style>
        /* Container principal */
        .appuz-report-box {
            display: flex;
            flex-direction: column;
            padding: 20px;
            border: 1px solid #ddd;
            border-radius: 6px;
            background-color: #f7f7f7;
            margin-top: 20px;
        }
        .appuz-report-header {
            display: flex;
            justify-content: space-between;
            margin-bottom: 20px;
        }
        .appuz-report-header div {
            font-size: 36px;
            font-weight: bold;
            text-align: center;
        }
        .appuz-report-header div span {
            font-size: 16px;
            display: block;
        }
        .appuz-report-content {
            display: flex;
            flex-wrap: wrap;
            gap: 20px;
        }
        .appuz-report-item {
            
            padding: 10px;
            border: 1px solid #ddd;
            border-radius: 6px;
            text-align: center;
           
        }
        .appuz-container {
            display: flex;
            flex-wrap: wrap;
            gap: 20px;
        }
        
        /* Bloco para Adicionar Sessão e Outros Dados Futuros */
        .appuz-box {
         overflow-x: auto;
    flex: 1 1 100%;
    padding: 20px;
    border: 1px solid #ddd;
    border-radius: 6px;
    background-color: #f7f7f7;
    box-sizing: border-box;
    margin-bottom: 20px; /* Espaçamento entre seções */
}
        
        /* Ajuste para dois blocos lado a lado */
        .appuz-flex {
            display: flex;
            flex-wrap: wrap;
            gap: 20px;
        }
        
        .appuz-half {
            flex: 1 1 48%;
        }

        /* Estilo dos botões */
        .button-primary {
    background-color: #007cba;
    border-color: #007cba;
    color: #ffffff;
    padding: 10px 20px;
    font-size: 16px;
    font-weight: bold;
    border-radius: 6px;
    transition: background-color 0.3s ease, box-shadow 0.3s ease;
    box-shadow: 0 4px 6px rgba(0, 0, 0, 0.1);
}
        .button-primary:hover {
    background-color: #005fa3;
    border-color: #005fa3;
    box-shadow: 0 6px 8px rgba(0, 0, 0, 0.15);
}

        /* Formulários */
        form label {
    font-weight: 600;
    margin-top: 10px;
    display: block;
    margin-bottom: 5px;
    font-size: 14px;
    color: #333;
}
        input[type="datetime-local"], input[type="date"], select, input[type="text"] {
    width: 100%;
    padding: 10px;
    border: 1px solid #ddd;
    border-radius: 6px;
    box-sizing: border-box;
    margin-bottom: 15px;
    font-size: 14px;
    background-color: #fff;
    transition: border-color 0.3s ease;
} 

    input[type="datetime-local"]:focus, input[type="date"]:focus, select:focus, input[type="text"]:focus {
    border-color: #007cba;
    outline: none;
}

        /* Tabelas */
        .widefat th, .widefat td {
            padding: 12px 10px;
            text-align: left;
            border-bottom: 1px solid #eee;
        }
        .widefat tr:nth-child(even) {
            background-color: #f9f9f9;
        }

        /* Responsividade */
        @media (max-width: 768px) {
        
        .appuz-box table {
            width: 100%;
            min-width: 100%;
        }

        .appuz-box th, .appuz-box td {
            padding: 10px;
            text-align: center;
            font-size: 14px;
        }

        .appuz-box th {
            background-color: #d9d9d9;
            color: #ffffff;
        }
        
        
            .appuz-half {
                flex: 1 1 100%;
            }
            .wrap {
                padding: 15px;
            }
            .appuz-report-content > div {
            width: 100% !important;
            flex: 1 1 100%;
        }
            .button-primary {
                font-size: 14px;
                padding: 6px 12px;
            }
            table.widefat, .form-table {
                display: block;
                overflow-x: auto;
                white-space: nowrap;
            }
            label {
                display: block;
                font-size: 14px;
            }
            h1, h2 {
                font-size: 20px;
                margin-top: 15px;
            }
            
        }
    </style>';
}
add_action('admin_head', 'appuz_enqueue_styles');



function appuz_save_new_session() {
    global $wpdb;

    if (isset($_POST['add_appointment'])) {
        $patient_id = sanitize_text_field($_POST['patient_id']);
        $session_date = sanitize_text_field($_POST['session_date']);

        // Obter dias de folga e horários indisponíveis
        $days_off = get_option('appuz_days_off', []);
        $unavailable_times = explode(',', get_option('appuz_unavailable_times', ''));

        $session_day = date('l', strtotime($session_date)); // Dia da semana da sessão
        $session_time = date('H:i', strtotime($session_date)); // Hora da sessão

        // Verificar se a data está em um dia de folga
        if (in_array(strtolower($session_day), $days_off)) {
            echo '<div class="notice notice-error"><p>' . __('Esse dia está indisponível para agendamentos.', 'appuz') . '</p></div>';
            return;
        }

        // Verificar se o horário está em um dos horários indisponíveis
        foreach ($unavailable_times as $time_range) {
            list($start, $end) = explode('-', $time_range);
            if ($session_time >= $start && $session_time < $end) {
                echo '<div class="notice notice-error"><p>' . __('Esse horário está indisponível para agendamentos.', 'appuz') . '</p></div>';
                return;
            }
        }

        // Verificar se o horário é uma hora cheia
        $session_minutes = date('i', strtotime($session_date));
        if ($session_minutes !== '00') {
            echo '<div class="notice notice-error"><p>' . __('Somente horários em horas cheias são permitidos (ex: 17:00, 18:00).', 'appuz') . '</p></div>';
            return;
        }

        // Verificar se o horário já está ocupado
        $existing_appointment = $wpdb->get_row($wpdb->prepare("SELECT * FROM {$wpdb->prefix}appuz_appointments WHERE session_date = %s", $session_date));
        if ($existing_appointment) {
            echo '<div class="notice notice-error"><p>' . __('Este horário já está reservado para outro paciente.', 'appuz') . '</p></div>';
        } else {
            $wpdb->insert($wpdb->prefix . 'appuz_appointments', [
                'patient_id' => $patient_id,
                'session_date' => $session_date,
                'status' => 'confirmed'
            ]);
            echo '<div class="notice notice-success"><p>' . __('Novo agendamento adicionado com sucesso!', 'appuz') . '</p></div>';
        }
    }
}
add_action('admin_notices', 'appuz_save_new_session');








// Função AJAX para obter horários disponíveis com base na data e nas configurações de folga
function appuz_get_available_times_ajax() {
    $selected_date = sanitize_text_field($_POST['date']);
    $available_times = appuz_get_available_times($selected_date);

    foreach ($available_times as $time) {
        echo '<option value="' . esc_attr($time) . '">' . esc_html($time) . '</option>';
    }
    wp_die();
}
add_action('wp_ajax_appuz_get_available_times', 'appuz_get_available_times_ajax');
add_action('wp_ajax_nopriv_appuz_get_available_times', 'appuz_get_available_times_ajax');



// Função para agendar a sessão automaticamente quando o status do pedido é "processando"
function appuz_schedule_appointment_on_processing($order_id) {
    global $wpdb;

    error_log("Função appuz_schedule_appointment_on_processing chamada para o pedido ID: " . $order_id);

    // Obter o pedido
    $order = wc_get_order($order_id);
    if (!$order) {
        error_log("Pedido não encontrado para o ID: " . $order_id);
        return;
    }

    $customer_id = $order->get_user_id();
    $patient_name = $order->get_billing_first_name() . ' ' . $order->get_billing_last_name();
    $patient_email = $order->get_billing_email();
    $patient_phone = $order->get_billing_phone();

    // Verificar se o paciente já está registrado
    $existing_patient = $wpdb->get_row($wpdb->prepare("SELECT id FROM {$wpdb->prefix}appuz_patients WHERE email = %s", $patient_email));
    if ($existing_patient) {
        $patient_id = $existing_patient->id;
        error_log("Paciente encontrado com ID: " . $patient_id);
    } else {
        // Registrar o paciente se não existir
        $inserted = $wpdb->insert($wpdb->prefix . 'appuz_patients', [
            'name' => $patient_name,
            'email' => $patient_email,
            'phone' => $patient_phone,
            'created_at' => current_time('mysql')
        ]);

        if ($inserted) {
            $patient_id = $wpdb->insert_id;
            error_log("Novo paciente registrado com ID: " . $patient_id);
        } else {
            error_log("Erro ao registrar o novo paciente.");
            return;
        }
    }

    // Recuperar as datas e horários de sessão do pedido
    $session_dates = get_post_meta($order_id, 'appuz_session_dates', true);
    if (empty($session_dates)) {
        error_log("Nenhuma data de sessão encontrada no pedido.");
        return;
    }

    // Iterar sobre cada sessão e registrar no banco de dados
    foreach ($session_dates as $session) {
        $session_date = sanitize_text_field($session['date']);
        $session_time = sanitize_text_field($session['time']);
        $session_datetime = $session_date . ' ' . $session_time;

        // Registrar a sessão na tabela de agendamentos
        $wpdb->insert($wpdb->prefix . 'appuz_appointments', [
            'patient_id' => $patient_id,
            'session_date' => $session_datetime,
            'status' => 'confirmed',
            'created_at' => current_time('mysql')
        ]);
        error_log("Sessão agendada para o paciente ID: $patient_id na data: $session_datetime");
    }
}
add_action('woocommerce_order_status_processing', 'appuz_schedule_appointment_on_processing');


// 1. Remover itens "Dashboard", "Orders" e "Downloads" e adicionar "Agendamentos" como padrão
function appuz_customize_account_menu_items($items) {
    unset($items['dashboard'], $items['orders'], $items['downloads'], $items['edit-address']); // Remove itens indesejados
    $items['appointments'] = __('Meus Agendamentos', 'appuz'); // Adiciona Agendamentos como o primeiro item
    return $items;
}
add_filter('woocommerce_account_menu_items', 'appuz_customize_account_menu_items');

// 2. Criar o endpoint "Agendamentos" para exibir a lista de agendamentos
function appuz_add_appointments_endpoint() {
    add_rewrite_endpoint('appointments', EP_ROOT | EP_PAGES);
}
add_action('init', 'appuz_add_appointments_endpoint');

// Estilo para exibir os agendamentos como cartões no desktop e mobile
function appuz_custom_appointments_card_styles() {
    echo '
    <style>
        .appuz-appointment-card {
            display: flex;
            flex-direction: column;
            gap: 10px;
            padding: 15px;
            margin-bottom: 15px;
            border-radius: 8px;
            box-shadow: 0 1px 3px rgba(0, 0, 0, 0.1);
            color: #fff;
            font-weight: bold;
        }
        .appuz-appointment-card.confirmed {
            background-color: #4CAF50; /* Verde para confirmado */
        }
        .appuz-appointment-card.pending {
            background-color: #FFC107; /* Amarelo para pendente */
        }
        .appuz-appointment-card.canceled {
            background-color: #F44336; /* Vermelho para cancelado */
        }
        .appuz-appointment-card div {
            padding: 10px;
            background-color: rgba(255, 255, 255, 0.2);
            border-radius: 4px;
            text-align: center;
        }
    </style>';
}
add_action('wp_head', 'appuz_custom_appointments_card_styles');

// Exibir os agendamentos do cliente na página "Meus Agendamentos" em formato de cartão
function appuz_display_customer_appointments() {
    $user_email = wp_get_current_user()->user_email;
    global $wpdb;

    // Buscar o ID do paciente na tabela de pacientes usando o e-mail do usuário
    $patient = $wpdb->get_row($wpdb->prepare(
        "SELECT id FROM {$wpdb->prefix}appuz_patients WHERE email = %s", 
        $user_email
    ));

    // Verificar se o paciente existe antes de buscar os agendamentos
    if ($patient) {
        $patient_id = $patient->id;

        // Listar agendamentos do cliente
        $appointments = $wpdb->get_results($wpdb->prepare(
            "SELECT * FROM {$wpdb->prefix}appuz_appointments WHERE patient_id = %d ORDER BY session_date ASC", 
            $patient_id
        ));

        echo '<h2>' . __('Meus Agendamentos', 'appuz') . '</h2>';

        if ($appointments) {
            foreach ($appointments as $appointment) {
                $data_hora = esc_html(date('d/m/Y', strtotime($appointment->session_date)));
                $horario = esc_html(date('H:i', strtotime($appointment->session_date)));

                // Define a classe de estilo com base no status do agendamento
                $status_class = '';
                $status_text = '';
                if ($appointment->status === 'confirmed') {
                    $status_class = 'confirmed';
                    $status_text = 'Confirmado';
                } elseif ($appointment->status === 'pending') {
                    $status_class = 'pending';
                    $status_text = 'Pendente';
                } elseif ($appointment->status === 'canceled') {
                    $status_class = 'canceled';
                    $status_text = 'Cancelado';
                }

                // Exibir o cartão de agendamento
                echo '<div class="appuz-appointment-card ' . $status_class . '">';
                echo '<div>' . __('Data do Agendamento:', 'appuz') . ' ' . $data_hora . '</div>';
                echo '<div>' . __('Horário do Agendamento:', 'appuz') . ' ' . $horario . '</div>';
                echo '<div>' . __('Status:', 'appuz') . ' ' . $status_text . '</div>';
                echo '</div>';
            }
        } else {
            echo '<p>' . __('Nenhum agendamento encontrado.', 'appuz') . '</p>';
        }
    } else {
        echo '<p>' . __('Nenhum agendamento encontrado.', 'appuz') . '</p>';
    }
}
add_action('woocommerce_account_appointments_endpoint', 'appuz_display_customer_appointments');

// Função para desativar cache na página do usuário
function appuz_disable_cache_for_user_pages() {
    if (is_account_page()) { // Verifica se é a página "Minha Conta" do WooCommerce
        // Cabeçalhos HTTP para desabilitar cache do navegador e proxies
        header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0');
        header('Cache-Control: post-check=0, pre-check=0', false);
        header('Pragma: no-cache');
        header('Expires: Wed, 11 Jan 1984 05:00:00 GMT');
    }
}
add_action('template_redirect', 'appuz_disable_cache_for_user_pages');

// Desativar cache de feeds e REST API para dados atualizados
function appuz_no_cache_headers($headers) {
    $headers['Cache-Control'] = 'no-store, no-cache, must-revalidate, max-age=0';
    $headers['Pragma'] = 'no-cache';
    return $headers;
}
add_filter('wp_headers', 'appuz_no_cache_headers');
add_filter('rest_cache_headers', 'appuz_no_cache_headers');

// Desabilitar cache em scripts e estilos para recarregar sempre a versão mais recente
function appuz_disable_script_style_cache($src) {
    return add_query_arg('v', time(), $src);
}
add_filter('style_loader_src', 'appuz_disable_script_style_cache', 10, 1);
add_filter('script_loader_src', 'appuz_disable_script_style_cache', 10, 1);

// Meta tags para desabilitar cache do lado do navegador
function appuz_no_cache_meta_tags() {
    if (is_account_page()) { // Apenas na página "Minha Conta"
        echo '<meta http-equiv="Cache-Control" content="no-store, no-cache, must-revalidate, max-age=0" />';
        echo '<meta http-equiv="Pragma" content="no-cache" />';
        echo '<meta http-equiv="Expires" content="0" />';
    }
}
add_action('wp_head', 'appuz_no_cache_meta_tags');

// Redirecionar para a página inicial ao fazer logout
function appuz_redirect_after_logout() {
    wp_redirect(home_url());
    exit;
}
add_action('wp_logout', 'appuz_redirect_after_logout');


// Redirecionar para "Meus Agendamentos" após o login
function appuz_redirect_after_login($redirect, $user) {
    // Verifica se o usuário está logado e é um cliente
    if (isset($user->roles) && is_array($user->roles) && in_array('customer', $user->roles)) {
        $redirect = wc_get_account_endpoint_url('appointments'); // Redireciona para o endpoint "Meus Agendamentos"
    }
    return $redirect;
}
add_filter('woocommerce_login_redirect', 'appuz_redirect_after_login', 10, 2);


// Função para cancelar, excluir ou atualizar uma sessão
function appuz_manage_session_actions() {
    global $wpdb;

    if (isset($_GET['action']) && isset($_GET['appointment_id'])) {
        $appointment_id = intval($_GET['appointment_id']);
        
        if ($_GET['action'] === 'cancel') {
            $wpdb->update(
                $wpdb->prefix . 'appuz_appointments',
                ['status' => 'canceled'],
                ['id' => $appointment_id]
            );
            echo '<div class="notice notice-success"><p>' . __('Sessão cancelada com sucesso.', 'appuz') . '</p></div>';
        } elseif ($_GET['action'] === 'delete') {
            $wpdb->delete(
                $wpdb->prefix . 'appuz_appointments',
                ['id' => $appointment_id]
            );
            echo '<div class="notice notice-success"><p>' . __('Sessão excluída com sucesso.', 'appuz') . '</p></div>';
        } elseif ($_POST['edit_appointment']) {
            $session_date = sanitize_text_field($_POST['session_date']);
            $status = sanitize_text_field($_POST['status']);
            $wpdb->update(
                $wpdb->prefix . 'appuz_appointments',
                [
                    'session_date' => $session_date,
                    'status' => $status,
                ],
                ['id' => $appointment_id]
            );
            echo '<div class="notice notice-success"><p>' . __('Sessão atualizada com sucesso.', 'appuz') . '</p></div>';
        }
    }
}
add_action('admin_notices', 'appuz_manage_session_actions');




// Página de Agendamentos no Admin
function appuz_admin_menu() {
    add_menu_page(
        __('Appuz Appointments', 'appuz') . ' ' . APPUZ_PLUGIN_VERSION,
        __('Appuz Appointments', 'appuz'),
        'manage_options',
        'appuz_appointments',
        'appuz_appointments_page',
        'dashicons-calendar-alt',
        6
    );
    
    add_submenu_page(
        'appuz_appointments',
        __('Configurar Disponibilidade', 'appuz'),
        __('Disponibilidade', 'appuz'),
        'manage_options',
        'appuz_availability',
        'appuz_availability_page'
    );
}
add_action('admin_menu', 'appuz_admin_menu');



// Adiciona uma subpágina oculta para editar a sessão
function appuz_add_edit_page() {
    add_submenu_page(
        null, // Define como "null" para ocultar no menu principal
        __('Editar Sessão', 'appuz'),
        __('Editar Sessão', 'appuz'),
        'manage_options',
        'appuz_edit_session',
        'appuz_edit_session_page'
    );
}
add_action('admin_menu', 'appuz_add_edit_page');


// Função para salvar novo paciente no banco de dados
function appuz_save_new_patient() {
    global $wpdb;

    if (isset($_POST['add_patient'])) {
        $name = sanitize_text_field($_POST['patient_name']);
        $email = sanitize_email($_POST['patient_email']);
        $phone = sanitize_text_field($_POST['patient_phone']);

        if (!empty($name) && !empty($email)) {
            $wpdb->insert($wpdb->prefix . 'appuz_patients', [
                'name' => $name,
                'email' => $email,
                'phone' => $phone
            ]);
            echo '<div class="notice notice-success"><p>' . __('Paciente adicionado com sucesso!', 'appuz') . '</p></div>';
        } else {
            echo '<div class="notice notice-error"><p>' . __('Por favor, preencha todos os campos obrigatórios.', 'appuz') . '</p></div>';
        }
    }
}
add_action('admin_notices', 'appuz_save_new_patient');


// Página de edição de sessão
function appuz_edit_session_page() {
    global $wpdb;

    // Verifica se existe um ID de sessão para editar
    if (isset($_GET['appointment_id'])) {
        $appointment_id = intval($_GET['appointment_id']);
        $appointment = $wpdb->get_row($wpdb->prepare("SELECT * FROM {$wpdb->prefix}appuz_appointments WHERE id = %d", $appointment_id));

        // Verifica se a sessão existe
        if (!$appointment) {
            echo '<div class="notice notice-error"><p>' . __('Sessão não encontrada.', 'appuz') . '</p></div>';
            return;
        }

        // Verifica se o formulário foi enviado
        if (isset($_POST['edit_appointment'])) {
            $session_date = sanitize_text_field($_POST['session_date']);
            $status = sanitize_text_field($_POST['status']);

            // Atualiza a sessão no banco de dados
            $wpdb->update(
                $wpdb->prefix . 'appuz_appointments',
                [
                    'session_date' => $session_date,
                    'status' => $status,
                ],
                ['id' => $appointment_id]
            );

            // Redireciona para a página principal com mensagem de sucesso
            wp_redirect(admin_url('admin.php?page=appuz_appointments&message=success'));
            exit;
        }

        // Exibe o formulário de edição com os dados da sessão
        echo '<div class="wrap">';
        echo '<h1>' . __('Editar Sessão', 'appuz') . '</h1>';
        echo '<form method="post" action="">';
        echo '<input type="hidden" name="appointment_id" value="' . esc_attr($appointment->id) . '">';

        echo '<p><label for="session_date">' . __('Data e Hora da Sessão:', 'appuz') . '</label>';
        echo '<input type="datetime-local" name="session_date" value="' . esc_attr(date('Y-m-d\TH:i', strtotime($appointment->session_date))) . '" required></p>';

        echo '<p><label for="status">' . __('Status:', 'appuz') . '</label>';
        echo '<select name="status">';
        echo '<option value="confirmed"' . selected($appointment->status, 'confirmed', false) . '>' . __('Confirmado', 'appuz') . '</option>';
        echo '<option value="pending"' . selected($appointment->status, 'pending', false) . '>' . __('Pendente', 'appuz') . '</option>';
        echo '<option value="canceled"' . selected($appointment->status, 'canceled', false) . '>' . __('Cancelado', 'appuz') . '</option>';
        echo '</select></p>';

        echo '<button type="submit" name="edit_appointment" class="button button-primary">' . __('Salvar Alterações', 'appuz') . '</button>';
        echo '</form>';
        echo '</div>';
    } else {
        echo '<p>' . __('ID da sessão não foi fornecido.', 'appuz') . '</p>';
    }
}


function appuz_manage_patient_actions() {
    global $wpdb;

    if (isset($_GET['action']) && isset($_GET['patient_id'])) {
        $patient_id = intval($_GET['patient_id']);

        // Excluir paciente
        if ($_GET['action'] === 'delete_patient') {
            $wpdb->delete("{$wpdb->prefix}appuz_patients", ['id' => $patient_id]);
            echo '<div class="notice notice-success"><p>' . __('Paciente excluído com sucesso.', 'appuz') . '</p></div>';
        }
    }
}
add_action('admin_notices', 'appuz_manage_patient_actions');




function appuz_edit_patient_form() {
    global $wpdb;

    if (isset($_GET['action']) && $_GET['action'] === 'edit_patient' && isset($_GET['patient_id'])) {
        $patient_id = intval($_GET['patient_id']);
        $patient = $wpdb->get_row($wpdb->prepare("SELECT * FROM {$wpdb->prefix}appuz_patients WHERE id = %d", $patient_id));

        if (!$patient) {
            echo '<div class="notice notice-error"><p>' . __('Paciente não encontrado.', 'appuz') . '</p></div>';
            return;
        }

        // Salvar alterações no paciente
        if (isset($_POST['save_patient'])) {
            $name = sanitize_text_field($_POST['patient_name']);
            $email = sanitize_email($_POST['patient_email']);
            $phone = sanitize_text_field($_POST['patient_phone']);

            $wpdb->update(
                "{$wpdb->prefix}appuz_patients",
                ['name' => $name, 'email' => $email, 'phone' => $phone],
                ['id' => $patient_id]
            );

            echo '<div class="notice notice-success"><p>' . __('Paciente atualizado com sucesso.', 'appuz') . '</p></div>';

            // Redireciona de volta para a página de pacientes
            wp_redirect(admin_url('admin.php?page=appuz_appointments'));
            exit;
        }

        // Formulário de edição de paciente
        echo '<div class="wrap">';
        echo '<h1>' . __('Editar Paciente', 'appuz') . '</h1>';
        echo '<form method="post" action="">';
        echo '<p><label for="patient_name">' . __('Nome:', 'appuz') . '</label>';
        echo '<input type="text" name="patient_name" value="' . esc_attr($patient->name) . '" required></p>';

        echo '<p><label for="patient_email">' . __('Email:', 'appuz') . '</label>';
        echo '<input type="email" name="patient_email" value="' . esc_attr($patient->email) . '" required></p>';

        echo '<p><label for="patient_phone">' . __('Telefone:', 'appuz') . '</label>';
        echo '<input type="tel" name="patient_phone" value="' . esc_attr($patient->phone) . '"></p>';

        echo '<button type="submit" name="save_patient" class="button button-primary">' . __('Salvar Alterações', 'appuz') . '</button>';
        echo '</form>';
        echo '</div>';
    }
}
add_action('admin_notices', 'appuz_edit_patient_form');

// Função para mover sessões com data passada para a lista de concluídas
function appuz_move_completed_sessions() {
    global $wpdb;
    $current_date = current_time('mysql');

    // Atualizar o status das sessões que já passaram para "completed"
    $wpdb->query($wpdb->prepare(
        "UPDATE {$wpdb->prefix}appuz_appointments 
        SET status = 'completed' 
        WHERE session_date < %s 
        AND status NOT IN ('completed', 'canceled')",
        $current_date
    ));
}

// Agendar a execução diária dessa função
if (!wp_next_scheduled('appuz_daily_move_completed_sessions')) {
    wp_schedule_event(time(), 'daily', 'appuz_daily_move_completed_sessions');
}
add_action('appuz_daily_move_completed_sessions', 'appuz_move_completed_sessions');

// Página de Gerenciamento de Agendamentos para o Terapeuta
function appuz_appointments_page() {
    
    global $wpdb;

    $filter_date = isset($_POST['filter_date']) ? sanitize_text_field($_POST['filter_date']) : '';
    $filter_status = isset($_POST['filter_status']) ? sanitize_text_field($_POST['filter_status']) : '';
    $where_clause = "1=1";
    if ($filter_date) {
        $where_clause .= " AND DATE(session_date) = '$filter_date'";
    }
    if ($filter_status) {
        $where_clause .= " AND status = '$filter_status'";
    }

    $appointments = $wpdb->get_results("SELECT a.*, p.name, p.email, p.phone FROM {$wpdb->prefix}appuz_appointments a
                                        JOIN {$wpdb->prefix}appuz_patients p ON a.patient_id = p.id
                                        WHERE $where_clause
                                        ORDER BY a.session_date ASC");

    $patients = $wpdb->get_results("SELECT * FROM {$wpdb->prefix}appuz_patients ORDER BY name ASC");
    // Lógica de contagem para o resumo de sessões
    $total_sessions = $wpdb->get_var("SELECT COUNT(*) FROM {$wpdb->prefix}appuz_appointments");
    $confirmed_sessions = $wpdb->get_var("SELECT COUNT(*) FROM {$wpdb->prefix}appuz_appointments WHERE status = 'confirmed'");
    $pending_sessions = $wpdb->get_var("SELECT COUNT(*) FROM {$wpdb->prefix}appuz_appointments WHERE status = 'pending'");
    $canceled_sessions = $wpdb->get_var("SELECT COUNT(*) FROM {$wpdb->prefix}appuz_appointments WHERE status = 'canceled'");
    $patients_count = $wpdb->get_var("SELECT COUNT(*) FROM {$wpdb->prefix}appuz_patients");

    // Sessões de hoje, amanhã e este mês
    $today = date('Y-m-d');
    $tomorrow = date('Y-m-d', strtotime('+1 day'));
    $this_month = date('Y-m');
    
    $today_sessions = $wpdb->get_var($wpdb->prepare("SELECT COUNT(*) FROM {$wpdb->prefix}appuz_appointments WHERE DATE(session_date) = %s", $today));
    $tomorrow_sessions = $wpdb->get_var($wpdb->prepare("SELECT COUNT(*) FROM {$wpdb->prefix}appuz_appointments WHERE DATE(session_date) = %s", $tomorrow));
    $month_sessions = $wpdb->get_var($wpdb->prepare("SELECT COUNT(*) FROM {$wpdb->prefix}appuz_appointments WHERE DATE_FORMAT(session_date, '%%Y-%%m') = %s", $this_month));

    echo '<div class="wrap">';
    echo '<h1>' . __('Olinda Santos Controle ', 'appuz') . APPUZ_PLUGIN_VERSION . '</h1>';

   // Estilos para organizar a exibição das seções
echo '<style>
    .appuz-container { display: flex; flex-wrap: wrap; gap: 20px; }
    .appuz-box { padding: 20px; border: 1px solid #ddd; border-radius: 6px; background-color: #f7f7f7; }
    .appuz-half { flex: 1 1 48%; } /* Define largura de 48% para cada caixa em uma linha */
    .appuz-full { flex: 1 1 100%; } /* Define largura de 100% para a seção de Filtros */
</style>';
// Box principal para o relatório
    echo '<div class="appuz-box">';

    // Header do relatório com sessões de hoje e amanhã
    echo '<div class="appuz-report-header">';
    echo '<div>' . esc_html($today_sessions) . '<span style=" padding-top: 15px; ">' . __('Sessões agendadas para hoje', 'appuz') . '</span></div>';
    echo '<div>' . esc_html($tomorrow_sessions) . '<span style=" padding-top: 15px; ">' . __('Sessões agendadas para amanhã', 'appuz') . '</span></div>';
    echo '</div>';

   // Conteúdo do relatório com os demais dados
echo '<div class="appuz-report-content" style="display: flex; flex-wrap: wrap; gap: 10px;">';
echo '<div style="background-color: #5A9BD5; color: #ffffff; padding: 10px; border-radius: 5px; text-align: center; width: 47%; flex: 1 1 47%; box-sizing: border-box;">';
echo '<strong>' . __('Sessões este mês:', 'appuz') . '</strong> ' . esc_html($month_sessions) . '</div>';

echo '<div style="background-color: #4CAF50; color: #ffffff; padding: 10px; border-radius: 5px; text-align: center; width: 47%; flex: 1 1 47%; box-sizing: border-box;">';
echo '<strong>' . __('Sessões Confirmadas:', 'appuz') . '</strong> ' . esc_html($confirmed_sessions) . '</div>';

echo '<div style="background-color: #9C27B0; color: #ffffff; padding: 10px; border-radius: 5px; text-align: center; width: 47%; flex: 1 1 47%; box-sizing: border-box;">';
echo '<strong>' . __('Número de Pacientes:', 'appuz') . '</strong> ' . esc_html($patients_count) . '</div>';

echo '<div style="background-color: #FFC107; color: #ffffff; padding: 10px; border-radius: 5px; text-align: center; width: 47%; flex: 1 1 47%; box-sizing: border-box;">';
echo '<strong>' . __('Sessões Pendentes:', 'appuz') . '</strong> ' . esc_html($pending_sessions) . '</div>';

echo '<div style="background-color: #F44336; color: #ffffff; padding: 10px; border-radius: 5px; text-align: center; width: 100%; flex: 1 1 100%; box-sizing: border-box;">';
echo '<strong>' . __('Sessões Canceladas:', 'appuz') . '</strong> ' . esc_html($canceled_sessions) . '</div>';
echo '</div>';

    echo '</div>'; // Fecha o box principal do relatório


// Container principal para as seções
echo '<div class="appuz-container">';

// Seção de Adicionar Sessão
echo '<div class="appuz-box appuz-half">';
echo '<h2>' . __('Adicionar Sessão', 'appuz') . '</h2>';
echo '<form method="post" action="">';
echo '<p><label for="patient_id">' . __('Paciente:', 'appuz') . '</label>';
echo '<select name="patient_id" required>';
echo '<option value="">' . __('Selecione um paciente', 'appuz') . '</option>';
foreach ($patients as $patient) {
    echo '<option value="' . esc_attr($patient->id) . '">' . esc_html($patient->name) . '</option>';
}
echo '</select></p>';

echo '<p><label for="session_date">' . __('Data e Hora da Sessão:', 'appuz') . '</label>';
echo '<input type="datetime-local" name="session_date" required></p>';

echo '<button type="submit" name="add_appointment" class="button button-primary">' . __('Adicionar Agendamento', 'appuz') . '</button>';
echo '</form>';
echo '</div>';

// Seção de Cadastro de Pacientes
echo '<div class="appuz-box appuz-half">';
echo '<h2>' . __('Cadastro de Pacientes', 'appuz') . '</h2>';
echo '<form method="post" action="">';
echo '<label for="patient_name">' . __('Nome:', 'appuz') . '</label>';
echo '<input type="text" name="patient_name" required>';

echo '<label for="patient_email">' . __('E-mail:', 'appuz') . '</label>';
echo '<input type="email" name="patient_email" required>';

echo '<label for="patient_phone">' . __('Telefone:', 'appuz') . '</label>';
echo '<input type="tel" name="patient_phone">';

echo '<button type="submit" name="add_patient" class="button button-primary">' . __('Adicionar Paciente', 'appuz') . '</button>';
echo '</form>';
echo '</div>';

// Seção de Filtros
echo '<div class="appuz-box appuz-half">';
echo '<h2>' . __('Filtros', 'appuz') . '</h2>';
echo '<form method="post" action="">';
echo '<label for="filter_date">' . __('Filtrar por Data:', 'appuz') . '</label> ';
echo '<input type="date" name="filter_date" value="' . esc_attr($filter_date) . '"> ';

echo '<label for="filter_status">' . __('Filtrar por Status:', 'appuz') . '</label> ';
echo '<select name="filter_status">';
echo '<option value="">' . __('Todos', 'appuz') . '</option>';
echo '<option value="confirmed"' . selected($filter_status, 'confirmed', false) . '>' . __('Confirmado', 'appuz') . '</option>';
echo '<option value="pending"' . selected($filter_status, 'pending', false) . '>' . __('Pendente', 'appuz') . '</option>';
echo '<option value="canceled"' . selected($filter_status, 'canceled', false) . '>' . __('Cancelado', 'appuz') . '</option>';
echo '</select> ';

echo '<button type="submit" class="button button-primary">' . __('Filtrar', 'appuz') . '</button>';
echo '</form>';
echo '</div>';




// Lista de Sessões
$current_datetime = current_time('mysql');

$appointments = $wpdb->get_results("
    SELECT a.*, p.name, p.email 
    FROM {$wpdb->prefix}appuz_appointments a
    JOIN {$wpdb->prefix}appuz_patients p 
    ON a.patient_id = p.id
    WHERE a.session_date >= '$current_datetime'
    ORDER BY a.session_date ASC
");

echo '<div class="appuz-box">';
echo '<h2>' . __('Lista de Sessões', 'appuz') . '</h2>';
echo '<table class="widefat fixed" cellspacing="0">';
echo '<thead><tr><th>' . __('Paciente', 'appuz') . '</th><th>' . __('Data e Hora', 'appuz') . '</th><th>' . __('Status', 'appuz') . '</th><th>' . __('Ações', 'appuz') . '</th></tr></thead>';
echo '<tbody>';

if (!empty($appointments)) {
    foreach ($appointments as $appointment) {
        // Mapeamento de status para tradução e estilos
        $status_map = [
            'confirmed' => ['text' => 'Confirmado', 'style' => 'background-color: #A5D6A7; color: #2E7D32; border-radius: 4px; text-align: center;'],
            'pending' => ['text' => 'Pendente', 'style' => 'background-color: #FFF59D; color: #F57F17; border-radius: 4px; text-align: center;'],
            'canceled' => ['text' => 'Cancelado', 'style' => 'background-color: #EF9A9A; color: #C62828; border-radius: 4px; text-align: center;']
        ];

        $status_text = isset($status_map[$appointment->status]) ? $status_map[$appointment->status]['text'] : $appointment->status;
        $status_style = isset($status_map[$appointment->status]) ? $status_map[$appointment->status]['style'] : '';

        echo '<tr>';
        echo '<td>' . esc_html($appointment->name) . ' (' . esc_html($appointment->email) . ')</td>';
        echo '<td>' . esc_html(date('d/m/Y H:i', strtotime($appointment->session_date))) . '</td>';
        echo '<td style="' . esc_attr($status_style) . '">' . esc_html($status_text) . '</td>';
        echo '<td>';
        echo '<a href="' . admin_url('admin.php?page=appuz_edit_session&appointment_id=' . $appointment->id) . '" class="button" style="margin-right: 5px;">' . __('Editar', 'appuz') . '</a>';
        echo '<a href="' . admin_url('admin.php?page=appuz_appointments&action=cancel&appointment_id=' . $appointment->id) . '" class="button" style="background-color: #FFA500; color: #ffffff; margin-right: 5px;">' . __('Cancelar', 'appuz') . '</a>';
        echo '<a href="' . admin_url('admin.php?page=appuz_appointments&action=delete&appointment_id=' . $appointment->id) . '" class="button button-danger" style="background-color: #F44336; color: #ffffff;" onclick="return confirm(\'' . __('Tem certeza que deseja excluir esta sessão?', 'appuz') . '\')">' . __('Excluir', 'appuz') . '</a>';
        echo '</td>';
        echo '</tr>';
    }
} else {
    echo '<tr><td colspan="4">' . __('Nenhuma sessão encontrada.', 'appuz') . '</td></tr>';
}

echo '</tbody>';
echo '</table>';
echo '</div>';
    

    // Exibição de sessões concluídas
function appuz_display_completed_sessions() {
    global $wpdb;

    $completed_sessions = $wpdb->get_results("SELECT a.*, p.name, p.email FROM {$wpdb->prefix}appuz_appointments a
                                              JOIN {$wpdb->prefix}appuz_patients p ON a.patient_id = p.id
                                              WHERE a.status = 'completed'
                                              ORDER BY a.session_date DESC");

    echo '<div class="appuz-box">';
    echo '<h2>' . __('Sessões Concluídas', 'appuz') . '</h2>';
    echo '<table class="widefat fixed" cellspacing="0">';
    echo '<thead><tr><th>' . __('Paciente', 'appuz') . '</th><th>' . __('Data e Hora', 'appuz') . '</th><th>' . __('Status', 'appuz') . '</th></tr></thead>';
    echo '<tbody>';

    foreach ($completed_sessions as $session) {
        echo '<tr>';
        echo '<td>' . esc_html($session->name) . ' (' . esc_html($session->email) . ')</td>';
        echo '<td>' . esc_html($session->session_date) . '</td>';
        echo '<td>' . __('Concluído', 'appuz') . '</td>';
        echo '</tr>';
    }

    echo '</tbody>';
    echo '</table>';
    echo '</div>';
}
    
    
    // Sessões Concluídas
$current_datetime = current_time('mysql');

$completed_appointments = $wpdb->get_results("
    SELECT a.*, p.name, p.email 
    FROM {$wpdb->prefix}appuz_appointments a
    JOIN {$wpdb->prefix}appuz_patients p 
    ON a.patient_id = p.id
    WHERE a.session_date < '$current_datetime'
    ORDER BY a.session_date DESC
");

echo '<div class="appuz-box">';
echo '<h2>' . __('Sessões Concluídas', 'appuz') . '</h2>';
echo '<table class="widefat fixed" cellspacing="0">';
echo '<thead><tr><th>' . __('Paciente', 'appuz') . '</th><th>' . __('Data e Hora', 'appuz') . '</th><th>' . __('Status', 'appuz') . '</th></tr></thead>';
echo '<tbody>';

if (!empty($completed_appointments)) {
    foreach ($completed_appointments as $appointment) {
        echo '<tr>';
        echo '<td>' . esc_html($appointment->name) . ' (' . esc_html($appointment->email) . ')</td>';
        echo '<td>' . esc_html(date('d/m/Y H:i', strtotime($appointment->session_date))) . '</td>';
        echo '<td>' . __('Concluído', 'appuz') . '</td>';
        echo '</tr>';
    }
} else {
    echo '<tr><td colspan="3">' . __('Nenhuma sessão concluída encontrada.', 'appuz') . '</td></tr>';
}

echo '</tbody>';
echo '</table>';
echo '</div>';

// Sessões Canceladas
$cancelled_appointments = $wpdb->get_results("
    SELECT a.*, p.name, p.email 
    FROM {$wpdb->prefix}appuz_appointments a
    JOIN {$wpdb->prefix}appuz_patients p 
    ON a.patient_id = p.id
    WHERE a.status = 'canceled'
    ORDER BY a.session_date ASC
");

echo '<div class="appuz-box">';
echo '<h2>' . __('Sessões Canceladas', 'appuz') . '</h2>';
echo '<table class="widefat fixed" cellspacing="0">';
echo '<thead><tr><th>' . __('Paciente', 'appuz') . '</th><th>' . __('Data e Hora', 'appuz') . '</th><th>' . __('Status', 'appuz') . '</th></tr></thead>';
echo '<tbody>';

foreach ($cancelled_appointments as $appointment) {
    echo '<tr>';
    echo '<td>' . esc_html($appointment->name) . ' (' . esc_html($appointment->email) . ')</td>';
    echo '<td>' . esc_html(date('d/m/Y H:i', strtotime($appointment->session_date))) . '</td>';
    echo '<td>' . __('Cancelado', 'appuz') . '</td>';
    echo '</tr>';
}

echo '</tbody>';
echo '</table>';
echo '</div>';

// Lista de Pacientes
echo '<div class="appuz-box">';
echo '<h2>' . __('Lista de Pacientes', 'appuz') . '</h2>';

// Recuperar todos os pacientes do banco de dados
$patients = $wpdb->get_results("SELECT * FROM {$wpdb->prefix}appuz_patients ORDER BY name ASC");

// Tabela de pacientes
echo '<table class="widefat fixed" cellspacing="0">';
echo '<thead><tr><th>' . __('Nome', 'appuz') . '</th><th>' . __('Email', 'appuz') . '</th><th>' . __('Telefone', 'appuz') . '</th><th>' . __('Ações', 'appuz') . '</th></tr></thead>';
echo '<tbody>';

foreach ($patients as $patient) {
    echo '<tr>';
    echo '<td>' . esc_html($patient->name) . '</td>';
    echo '<td>' . esc_html($patient->email) . '</td>';
    echo '<td>' . esc_html($patient->phone) . '</td>';
    echo '<td>';
    echo '<a href="' . admin_url('admin.php?page=appuz_appointments&action=edit_patient&patient_id=' . $patient->id) . '" class="button">' . __('Editar', 'appuz') . '</a> ';
    echo '<a href="' . admin_url('admin.php?page=appuz_appointments&action=delete_patient&patient_id=' . $patient->id) . '" class="button button-danger" onclick="return confirm(\'' . __('Tem certeza que deseja excluir este paciente?', 'appuz') . '\')">' . __('Excluir', 'appuz') . '</a>';
    echo '</td>';
    echo '</tr>';
}

echo '</tbody>';
echo '</table>';
echo '</div>';
    
}
